'****************************************************************************
'*  Name    : GameShield.BAS                                                *
'*  Author  : Gavin Wiggett                                                 *
'*  Notice  : Copyright (c) 2013 Bitfogav                                   *
'*          : All Rights Reserved                                           *
'*  Date    : 20/07/2013                                                    *
'*  Version : 1.0                                                           *
'*  Notes   : Contains - software based shift functions for the 74595       *
'*          : - Keypress to read key buttons                                *
'*          : - ability to clear the matrix display                         *
'*          : - Init to setup pins for 74595 control                        *
'****************************************************************************

Module GameShield

' pin structure...
Structure PinType
   Dim Address As UShort
   Dim Pin As ushort
   Dim PinMask As ushort
End Structure  

' matrix data...
structure _matrix
   dim Anodes(8) as byte
   dim RedData(8) as byte
   dim GreenData(8) as byte 
   dim OrangeData(8) as byte
end structure

' pin config...
Dim _dataOut As PinType
dim _dataLatch as PinType
Dim _dataClock As PinType 

' create a structure of matrix data...
public dim MatrixData as _matrix              

'****************************************************************************
'* Name    : ToggleClock                                                    *
'* Purpose : Toggle the clock pin.                                          *
'****************************************************************************
private Inline Sub ToggleClock()
   *(waddr1) = *(waddr1) Xor _dataClock.Pin
   DelayuS(2) 
   *(waddr1) = *(waddr1) Xor _dataClock.Pin
End Sub 

'****************************************************************************
'* Name    : ToggleLatch                                                    *
'* Purpose : Toggle the latch pin to store the shifted data.                *
'****************************************************************************
Private Inline Sub ToggleLatch()
   *(waddr1) = *(waddr1) Xor _dataLatch.Pin
   DelayuS(2)  
   *(waddr1) = *(waddr1) Xor _dataLatch.Pin
End Sub 

'****************************************************************************
'* Name    : MakeOutput                                                     *
'* Purpose : Make the data pin an output.                                   *
'****************************************************************************
private Inline Sub MakeOutput()
   waddr1 = _dataOut.Address - 4 
   *(waddr1) = *(waddr1) And _dataOut.PinMask
End Sub 

'****************************************************************************
'* Name    : MakeLatchOutput                                                *
'* Purpose : Make the latch pin an output.                                  *
'****************************************************************************
private Inline Sub MakeLatchOutput()
   waddr1 = _dataLatch.Address - 4 
   *(waddr1) = *(waddr1) And _dataLatch.PinMask
End Sub     

'****************************************************************************
'* Name    : LoadDataOutAndClockAddr                                        *
'* Purpose : Load the data and clock pin address into waddr(x).             *
'****************************************************************************
private Inline Sub LoadDataOutAndClockAddr()
   waddr0 = _dataOut.Address
   waddr1 = _dataClock.Address
End Sub   

'****************************************************************************
'* Name    : SetOutput                                                      *
'* Purpose : Sets the data out pin.                                         *
'****************************************************************************
Private Sub SetOutput(ByRef dataPin As Bit)
   _dataOut.Address = AddressOf(dataPin) + 2 
   _dataOut.Pin = BitOf(dataPin)
   _dataOut.PinMask = Not _dataOut.Pin
End Sub

'****************************************************************************
'* Name    : SetLatch                                                       *
'* Purpose : Sets the latch pin.                                            *
'****************************************************************************
Private Sub SetLatch(ByRef latchPin As Bit)
   _dataLatch.Address = AddressOf(latchPin) + 2 
   _dataLatch.Pin = BitOf(latchPin)
   _dataLatch.PinMask = Not _dataLatch.Pin
End Sub

'****************************************************************************
'* Name    : SetClock                                                       *
'* Purpose : Sets the clock pin. An optional idle high determines if the    *
'*         : clock pin idles high or low. The default is to idle low.       *
'****************************************************************************
Private Sub SetClock(ByRef clockPin As Bit, Optional idleHigh As Boolean = False)
   _dataClock.Address = AddressOf(clockPin) + 2  
   _dataClock.Pin = BitOf(clockPin)
   _dataClock.PinMask = Not _dataClock.Pin  
   waddr1 = _dataClock.Address
   If idleHigh Then
      *(waddr1) = *(waddr1) Or _dataClock.Pin
   Else
      *(waddr1) = *(waddr1) And _dataClock.PinMask   
   End If
   waddr1 = waddr1 - 4 
   *(waddr1) = *(waddr1) And _dataClock.PinMask
End Sub

'****************************************************************************
'* Name    : Out                                                            *
'* Purpose : Shift out a value by number of bits.                           *
'****************************************************************************
Private Sub Out(data As byte, Optional shiftAmount As Byte = 8)
   MakeOutput()
   MakeLatchOutput()
   LoadDataOutAndClockAddr()
   While shiftAmount > 0 
      If data.0 = 0 Then
         *(waddr0) = *(waddr0) And _dataOut.PinMask
      Else
         *(waddr0) = *(waddr0) Or _dataOut.Pin
      End If  
      ToggleClock()
      data = data >> 1
      shiftAmount -= 1 
   End While
   ClearWDT
End Sub

'****************************************************************************
'* Name    : ClearMatrix                                                    *
'* Purpose : Clears the Matrix.                                             *
'****************************************************************************
public sub ClearMatrix()
   Out(&H00)      ' green data
   Out(&H00)      ' red data
   Out(&H00)      ' anodes data
   ToggleLatch()  ' latch data in registers
end sub

'****************************************************************************
'* Name    : InitAnodes                                                     *
'* Purpose : Used to set the anodes for the matrix.                         *
'* Note    : This is important for the matrix to work correctly.            *
'****************************************************************************
private inline sub InitAnodes()
   dim hexVal as byte = &B00000001
   for i as byte = 0 to ubound(MatrixData.Anodes)  
      MatrixData.Anodes(i) = hexVal
      hexVal = hexVal << 1 
   next
end sub

'****************************************************************************
'* Name    : ClearRedData                                                   *
'* Purpose : Clears the Red Data array.                                     *
'****************************************************************************
public inline sub ClearRedData()
   erase(MatrixData.RedData)
end sub

'****************************************************************************
'* Name    : ClearGreenData                                                 *
'* Purpose : Clears the Green Data array.                                   *
'****************************************************************************
public inline sub ClearGreenData()
   erase(MatrixData.GreenData)
end sub

'****************************************************************************
'* Name    : ClearOrangeData                                                *
'* Purpose : Clears the Orange Data array.                                  *
'****************************************************************************
public inline sub ClearOrangeData()
   erase(MatrixData.OrangeData)
end sub

'****************************************************************************
'* Name    : UpdateMatrix                                                   *
'* Purpose : Sends out all data to matrix 74595's.                          *
'* Note    : should be polled in main code.                                 *
'****************************************************************************
Public sub UpdateMatrix()
   for i as byte = 0 to 7                     
      for x as byte = 0 to 10  
         Out(MatrixData.GreenData(i) or MatrixData.OrangeData(i))  
         Out(MatrixData.RedData(i) or MatrixData.OrangeData(i))        
         Out(MatrixData.Anodes(i))    
         ToggleLatch()               
      next                      
   next
end sub

'****************************************************************************
'* Name    : Keypress                                                       *
'* Purpose : Get pressed key/button                                         *
'* Note    : return key number, if 0 then no key/button was pressed.        *
'**************************************************************************** 
public function Keypress() as byte
   dim voltage as ushort = Adc.Read(A0) 
   Keypress = 0                         
   select voltage
      case < 50  : Keypress = 1  ' select
      case < 200 : Keypress = 2  ' fire
      case < 400 : Keypress = 3  ' right 
      case < 600 : Keypress = 4  ' up
      case < 800 : Keypress = 5  ' down
      case < 1000 : Keypress = 6 ' left 
   end select
end function

'******************************************************************************
'* Name    : Main                                                             *
'* Purpose : Configure the Matrix/Game Shield module before use.              *
'******************************************************************************      
Private sub Main()
   SetOutput(D7)
   SetLatch(D6)
   SetClock(D5) 
   InitAnodes()
   ClearMatrix()
end sub

End Module
