/*
 * USB HID bootloader for PIC32 microcontroller.
 *
 * Based on Microchip sources.
 * Heavily rewritten by Serge Vakulenko, <serge@vak.ru>.
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the 'Company') for its PIC(R) Microcontroller is intended and
 * supplied to you, the Company's customer, for use solely and
 * exclusively on Microchip PIC Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN 'AS IS' CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 */
#include "C:\Programming\Borland\Firewing\Runtime\Firewing Board\loaderFirmware\R2\32MX150F128\pic32mx.h"
#include "C:\Programming\Borland\Firewing\Runtime\Firewing Board\loaderFirmware\R2\32MX150F128\io.h"

/*
 * Flash memory.
 */
#define FLASH_BASE          0x1D000000  /* physical */
#define FLASH_BASE_END      (FLASH_BASE + 0x20000)
#define CPU_KHZ             40000

#define FLASH_USER       FLASH_BASE + 0x1000  /* physical: beginning of user application */

#define FLASH_OFFSET    0x80000000

#define FLASH_JUMP       (FLASH_USER + FLASH_OFFSET)

#define FLASH_PAGESZ        1024        /* bytes per page, for erasing */
#define FLASH_ROWSZ         128

#define BAUDRATE 38400

#define INIT_TIMEOUT 5000000
#define COMMS_TIMEOUT 5000000

PIC32_DEVCFG (
    0x1FFFFFFB,
    0xFF79CF59,
    0xFF790F59,
    0x0FFFFFFB
);

#define DEVICE_IS_32MX12    0xB1
#define FIRMWARE_VERSION    0x10

#define LOADER_CONNECT      0x80
#define LOADER_ACK          0x80
#define LOADER_GO           0x81
#define LOADER_ERROR_CRC    0x8A
#define LOADER_ERROR_VERIFY 0x8B
#define LOADER_ERROR_BADCOM     0x8C
#define LOADER_ERROR_NOTERASED  0x8D
#define LOADER_ERROR_BADADDRESS 0x8E

#define CMD_EEPROM_READ     0x01
#define CMD_EEPROM_WRITE    0x02
#define CMD_ROM_ERASE       0x04
#define CMD_ROM_READ        0x08
#define CMD_ROM_WRITE       0x10
#define CMD_BAUDRATE        0x20


unsigned int buffer[FLASH_ROWSZ / 4];

/*
 * GPIO pin control.
 */
#define TRIS_VAL(p)     (&p)[0]
#define TRIS_CLR(p)     (&p)[1]
#define TRIS_SET(p)     (&p)[2]
#define TRIS_INV(p)     (&p)[3]
#define PORT_VAL(p)     (&p)[4]
#define PORT_CLR(p)     (&p)[5]
#define PORT_SET(p)     (&p)[6]
#define PORT_INV(p)     (&p)[7]
#define LAT_VAL(p)      (&p)[8]
#define LAT_CLR(p)      (&p)[9]
#define LAT_SET(p)      (&p)[10]
#define LAT_INV(p)      (&p)[11]

#define PIC32_BRG_BAUD(fr,bd)   ((((fr)/8 + (bd)) / (bd) / 2) - 1)



/*
 * Boot code.
 */
asm ("          .section .startup,\"ax\",@progbits");
asm ("          .globl _start");
asm ("          .type _start, function");
asm ("_start:   la      $sp, _estack");
asm ("          la      $ra, main");
asm ("          la      $gp, _gp");
asm ("          jr      $ra");
asm ("          .text");

unsigned char crc;

void unlock() {
    mips_intr_disable();

    if (! (DMACON & 0x1000)) {
        DMACONSET = 0x1000;
        while (DMACON & 0x800)
            continue;
    }
    SYSKEY = 0;
    SYSKEY = 0xaa996655;
    SYSKEY = 0x556699aa;
    mips_intr_enable();
}

void lock() {
    mips_intr_disable();

    if (! (DMACON & 0x1000)) {
        DMACONSET = 0x1000;
        while (DMACON & 0x800)
            continue;
    }
    SYSKEY = 0;
    SYSKEY = 0x33333333;
    mips_intr_enable();
}

void led_init()
{
    TRIS_CLR(TRISB) = 1<<14;
    LAT_CLR(TRISB) = 1<<14;
}

void led_deinit()
{
    TRIS_SET(TRISB) = 1<<14;
    LAT_CLR(TRISB) = 1<<14;
}

void led_toggle()
{
    LAT_INV(TRISB) = 1<<14;
}

void led_on() {
    LAT_SET(TRISB) = 1<<14;
}

void led_off() {
    LAT_CLR(TRISB) = 1<<14;
}

void uart_init()
{

    unlock();
    CFGCONCLR = PIC32_CFGCON_IOLOCK;
    lock();
    
    U1RXR = 0b0001; // B6
    RPB7R = 0b0001; // U1TX
    TRIS_CLR(TRISB) = 1<<7;
    TRIS_SET(TRISB) = 1<<6;

    /*
     * Setup UART registers.
     * Compute the divisor for 38400 baud.
     */
    U1BRG = PIC32_BRG_BAUD (CPU_KHZ * 1000, BAUDRATE);
    U1STA = 0;

    U1MODESET = PIC32_UMODE_ON;

    U1MODE = PIC32_UMODE_PDSEL_8NPAR |	/* 8-bit data, no parity */
             PIC32_UMODE_ON;		/* UART Enable */
    U1STASET = PIC32_USTA_URXEN |	/* Receiver Enable */
               PIC32_USTA_UTXEN;	/* Transmit Enable */

    U1STA |= (PIC32_USTA_UTXEN | PIC32_USTA_URXEN);
}

void uart_putc (unsigned char c)
{
    while (U1STA & PIC32_USTA_UTXBF) 
        continue;
    U1TXREG = c;
}

int uart_getc ()
{
    if (!(U1STA & PIC32_USTA_URXDA))
        return -1;
    return U1RXREG;
}

void uart_puts(char *s) {
    char *p = s;
    while (*p) {
        uart_putc(*(p++));
    }
}

/*
 * Reset the microrontroller.
 */
void soft_reset()
{
    unlock();

    RSWRSTSET = 1;
    (void) RSWRST;

    for (;;)
        continue;
}

/*
 * Microsecond delay routine for MIPS processor.
 */
void udelay (unsigned usec)
{
    unsigned now = mips_read_c0_register (C0_COUNT, 0);
    unsigned final = now + usec * (CPU_KHZ / 1000);

    for (;;) {
        now = mips_read_c0_register (C0_COUNT, 0);

        /* This comparison is valid only when using a signed type. */
        if ((int) (now - final) >= 0)
            break;
    }
}

/*
 * Clear an array.
 */
void memzero (void *data, unsigned nbytes)
{
    unsigned *wordp = (unsigned*) data;
    unsigned nwords = nbytes / sizeof(int);

    while (nwords-- > 0)
        *wordp++ = 0;
}

/*
 * Copy an array.
 */
void memcopy (void *from, void *to, unsigned nbytes)
{
    unsigned *src = (unsigned*) from;
    unsigned *dst = (unsigned*) to;
    unsigned nwords = nbytes / sizeof(int);

    while (nwords-- > 0)
        *dst++ = *src++;
}

/*
 * BlinkUSBStatus turns on and off LEDs
 * corresponding to the USB device state.
 */
unsigned int led_count = 0;
void led_blink()
{

    led_count++;
    if (led_count >= 1024) {
        led_toggle();
        led_count = 0;
    }
}

/*
 * Perform non-volatile memory operation.
 */
void nvm_operation (unsigned op, unsigned address, unsigned data)
{
    int x;

    // Convert virtual address to physical
    NVMADDR = address & 0x1fffffff;
    if (op == PIC32_NVMCON_ROW_PGM) {
        NVMSRCADDR = data;
    } else {
        NVMDATA = data;
    }

    // Disable interrupts
    x = mips_intr_disable();

    // Enable Flash Write/Erase Operations
    NVMCON = (PIC32_NVMCON_WREN | op);

    // Data sheet prescribes 6us delay for LVD to become stable.
    // To be on the safer side, we shall set 7us delay.
    udelay (7);

    NVMKEY = 0xAA996655;
    NVMKEY = 0x556699AA;
    NVMCONSET = PIC32_NVMCON_WR; 

    // Wait for WR bit to clear
    while (NVMCON & PIC32_NVMCON_WR)
        continue;

    // Disable Flash Write/Erase operations
    NVMCONCLR = PIC32_NVMCON_WREN;

    // Enable interrupts
    mips_intr_restore (x);
}

void boot_user_program() {
    led_deinit();
    void (*fptr)(void) = (void (*)(void)) FLASH_JUMP;
    fptr();
    for (;;)
        continue;
}

unsigned char read_with_timeout(unsigned int attempts) {
    int res = -1;
    while (res == -1 && attempts > 0) {
        res = uart_getc();
        attempts--;
    }
    if (res == -1) {
        boot_user_program();
    }
    crc += (res & 0xFF);
    return res & 0xFF; 
}

unsigned int readAddress() {
    return ( 
        (read_with_timeout(COMMS_TIMEOUT) << 24) |
        (read_with_timeout(COMMS_TIMEOUT) << 16) |
        (read_with_timeout(COMMS_TIMEOUT) << 8) |
        (read_with_timeout(COMMS_TIMEOUT) << 0) 
    );
}

/*
 * Main program entry point.
 */
int main()
{
    unsigned char *chptr;
    unsigned int *iptr;
    unsigned int i;

    /* Initialize STATUS register: master interrupt disable. */
    mips_write_c0_register (C0_STATUS, 0, ST_CU0 | ST_BEV);

    /* Clear CAUSE register: use special interrupt vector 0x200. */
    mips_write_c0_register (C0_CAUSE, 0, CA_IV);

    /*
     * Setup wait states.
     */
//    CHECON = 2;
    BMXCONCLR = 0x40;
//    CHECONSET = 0x30;

    /* Disable JTAG port, to use it for i/o. */
    //DDPCON = 0;

    /* Config register: enable kseg0 caching. */
    mips_write_c0_register (C0_CONFIG, 0,
        mips_read_c0_register (C0_CONFIG, 0) | 3);
    mips_ehb();

    /* Initialize all .bss variables by zeros. */
    extern unsigned char __bss_start, __bss_end;
    memzero (&__bss_start, &__bss_end - &__bss_start);

    RCONCLR = PIC32_RCON_SWR; 

    uart_init();

    int initKey = read_with_timeout(INIT_TIMEOUT);
    if (initKey != LOADER_CONNECT) {
        boot_user_program();
    }
    uart_putc(LOADER_CONNECT);

    int goKey = read_with_timeout(COMMS_TIMEOUT);
    if (goKey != LOADER_GO) {
        boot_user_program();
    }

    uart_putc(FIRMWARE_VERSION);
    uart_putc((BAUDRATE >> 16) & 0xFF);
    uart_putc((BAUDRATE >> 8) & 0xFF);
    uart_putc((BAUDRATE) & 0xFF);
    uart_putc(DEVICE_IS_32MX12);
    uart_putc((DEVID >> 8) & 0xFF);
    uart_putc((DEVID) & 0xFF);

    uart_putc((FLASH_BASE >> 24) & 0xFF);
    uart_putc((FLASH_BASE >> 16) & 0xFF);
    uart_putc((FLASH_BASE >> 8) & 0xFF);
    uart_putc((FLASH_BASE) & 0xFF);

    uart_putc((FLASH_BASE_END >> 24) & 0xFF);
    uart_putc((FLASH_BASE_END >> 16) & 0xFF);
    uart_putc((FLASH_BASE_END >> 8) & 0xFF);
    uart_putc((FLASH_BASE_END) & 0xFF);

//    uart_putc(0xFF);
//    uart_putc(0xFF);
//    uart_putc(0xFF);

    uart_putc((FLASH_PAGESZ >> 8) & 0xFF);
    uart_putc((FLASH_PAGESZ) & 0xFF);
    uart_putc(FLASH_ROWSZ);
//    uart_putc(LOADER_ACK);

    // Ok, so we must be wanting to use the bootloader now.  We can start
    // fiddling with IO ports from this point on - didn't want to do it too
    // soon incase it inadvertantly turned on a mincing machine and killed
    // half the population of Kuala Lumpur.

    led_init();

    int address;

    for (;;) {

        int command = read_with_timeout(COMMS_TIMEOUT);
        crc = 0;
        if (command != -1) {
            switch (command) {
                case CMD_BAUDRATE:
                    //address = readAddress();
                    uart_putc(LOADER_ACK);
                    break;

                case CMD_ROM_ERASE:
                    address = readAddress();
                    if ((address < 0x1D000000) || (address > (0x1D000000 + 131072))) {
                        uart_putc(LOADER_ERROR_BADADDRESS);
                        break;
                    } 
                    nvm_operation(PIC32_NVMCON_PAGE_ERASE, address, 0);
                    uart_putc(LOADER_ACK);
                    break;

                case CMD_ROM_READ:
                    address = readAddress();
                    if ((address < 0x1D000000) || (address > (0x1D000000 + 131072))) {
                        uart_putc(LOADER_ERROR_BADADDRESS);
                        break;
                    } 
                    chptr = (unsigned char *)address;
                    uart_putc(*chptr);
                    uart_putc(LOADER_ACK);
                    break;

                case CMD_ROM_WRITE:
                    address = readAddress();
                    if ((address < 0x1D000000) || (address > (0x1D000000 + 131072))) {
                        uart_putc(LOADER_ERROR_BADADDRESS);
                        break;
                    } 
                    for (i=0; i < FLASH_ROWSZ/4; i ++) {
                        buffer[i] = 
                            read_with_timeout(COMMS_TIMEOUT) << 24 |
                            read_with_timeout(COMMS_TIMEOUT) << 16 |
                            read_with_timeout(COMMS_TIMEOUT) << 8 |
                            read_with_timeout(COMMS_TIMEOUT);
                    }
                    read_with_timeout(COMMS_TIMEOUT);
                    if (crc != 0) {
                        uart_putc(LOADER_ERROR_CRC);
                        break;
                    }
            
                    iptr = (unsigned int *)(FLASH_OFFSET + address);
                    for (i=0; i < FLASH_ROWSZ / 4; i++) {
                        if (iptr[i] != 0xFFFFFFFF) {
                            uart_putc(LOADER_ERROR_NOTERASED);
                            break;
                        }
                    }

//                        nvm_operation(PIC32_NVMCON_ROW_PGM, address, (unsigned int)buffer);

                    for (i=0; i < FLASH_ROWSZ/4; i++) {
                        nvm_operation(PIC32_NVMCON_WORD_PGM, address + (i*4), buffer[i]);
                    }

                    led_toggle();



                    for (i=0; i < FLASH_ROWSZ/4; i++) {
                        if (iptr[i] != buffer[i]) {
                            uart_putc(LOADER_ERROR_VERIFY);
                            uart_putc((iptr[i] >> 24) & 0xFF);
                            uart_putc((iptr[i] >> 16) & 0xFF);
                            uart_putc((iptr[i] >> 8) & 0xFF);
                            uart_putc(iptr[i] & 0xFF);
                            uart_putc((buffer[i] >> 24) & 0xFF);
                            uart_putc((buffer[i] >> 16) & 0xFF);
                            uart_putc((buffer[i] >> 8) & 0xFF);
                            uart_putc(buffer[i] & 0xFF);
                            break;
                        }
                    }

                    uart_putc(LOADER_ACK);
                    break;
                default:
                    uart_putc(LOADER_ERROR_BADCOM);
                    break;
            }

        } else {
            uart_puts("Comms timeout\r\n");
            boot_user_program();
        }
    }
}
